import React from 'react';
import { motion } from 'framer-motion';
import { MapPin, Shield, Smartphone, ArrowRight, Car, Bike, Truck } from 'lucide-react';
import { companyInfo } from '../data/mock';

const vehicleImages = {
  car: "https://images.unsplash.com/photo-1503376780353-7e6692767b70?crop=entropy&cs=srgb&fm=jpg&ixid=M3w4NjA1NzB8MHwxfHNlYXJjaHw0fHxjYXJ8ZW58MHx8fHwxNzY5MTkxNjg4fDA&ixlib=rb-4.1.0&q=85",
  moto: "https://images.unsplash.com/photo-1591637333184-19aa84b3e01f?crop=entropy&cs=srgb&fm=jpg&ixid=M3w4NjA1Mjh8MHwxfHNlYXJjaHwxfHxtb3RvcmN5Y2xlfGVufDB8fHx8MTc2OTE5MTY5M3ww&ixlib=rb-4.1.0&q=85",
  truck: "https://images.unsplash.com/photo-1592838064575-70ed626d3a0e?crop=entropy&cs=srgb&fm=jpg&ixid=M3w3NDk1Nzd8MHwxfHNlYXJjaHwzfHx0cnVja3xlbnwwfHx8fDE3NjkxOTE3MDB8MA&ixlib=rb-4.1.0&q=85"
};

const Hero = () => {
  return (
    <section id="inicio" className="relative min-h-screen flex items-center pt-20 overflow-hidden">
      {/* Background Effects */}
      <div className="absolute inset-0 bg-[var(--bg-primary)]">
        {/* Grid Pattern */}
        <div 
          className="absolute inset-0 opacity-[0.03]"
          style={{
            backgroundImage: `linear-gradient(var(--border-primary) 1px, transparent 1px), linear-gradient(90deg, var(--border-primary) 1px, transparent 1px)`,
            backgroundSize: '50px 50px'
          }}
        />
        
        {/* Glow Effects */}
        <div className="absolute top-1/4 left-1/4 w-96 h-96 bg-[var(--accent-primary)] opacity-5 rounded-full blur-[120px]" />
        <div className="absolute bottom-1/4 right-1/4 w-96 h-96 bg-[var(--accent-cyan)] opacity-5 rounded-full blur-[120px]" />
      </div>

      <div className="container mx-auto px-6 relative z-10">
        <div className="grid lg:grid-cols-2 gap-12 items-center">
          {/* Left Content */}
          <motion.div
            initial={{ opacity: 0, x: -50 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.7 }}
          >
            {/* Company Brand Highlight */}
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.1 }}
              className="flex items-center gap-5 mb-8"
            >
              <div className="relative">
                <div className="absolute inset-0 bg-[var(--accent-primary)] opacity-20 blur-xl rounded-full" />
                <img
                  src={companyInfo.logo}
                  alt={companyInfo.name}
                  className="h-24 md:h-32 w-auto relative z-10 drop-shadow-2xl"
                />
              </div>
              <div className="flex flex-col">
                <h2 className="text-3xl md:text-4xl font-black leading-none">
                  <span className="text-[var(--text-primary)]">Castanhal</span>
                </h2>
                <h2 className="text-3xl md:text-4xl font-black leading-none text-neon">
                  Tracking
                </h2>
                <span className="text-sm md:text-base text-[var(--text-muted)] uppercase tracking-[0.2em] mt-1">
                  Rastreamento Veicular
                </span>
              </div>
            </motion.div>

            {/* Badge */}
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.2 }}
              className="inline-flex items-center gap-2 px-4 py-2 rounded-full bg-[var(--accent-bg)] border border-[var(--accent-primary)] mb-6"
            >
              <span className="w-2 h-2 rounded-full bg-[var(--accent-primary)] animate-pulse" />
              <span className="text-[var(--accent-primary)] text-sm font-medium">
                {companyInfo.location}
              </span>
            </motion.div>

            <h1 className="text-3xl md:text-4xl lg:text-5xl font-bold leading-tight mb-6">
              Proteção Total para seu{' '}
              <span className="text-neon">Veículo</span>
            </h1>

            <p className="text-lg text-[var(--text-secondary)] mb-8 max-w-xl">
              Rastreamento veicular com localização em tempo real, bloqueio remoto e muito mais. 
              Equipamento gratuito, sem fidelidade e instalação a domicílio.
            </p>

            {/* Price Highlight */}
            <motion.div
              initial={{ opacity: 0, scale: 0.9 }}
              animate={{ opacity: 1, scale: 1 }}
              transition={{ delay: 0.4 }}
              className="inline-flex items-baseline gap-2 mb-8"
            >
              <span className="text-[var(--text-muted)] text-lg">A partir de</span>
              <span className="text-5xl font-bold text-neon">R$ {companyInfo.monthlyPrice}</span>
              <span className="text-[var(--text-muted)] text-lg">/mês</span>
            </motion.div>

            {/* CTAs */}
            <div className="flex flex-wrap gap-4 mb-12">
              <a
                href={companyInfo.whatsappLink}
                target="_blank"
                rel="noopener noreferrer"
                className="btn-primary"
              >
                Quero Rastrear Meu Veículo
                <ArrowRight size={20} />
              </a>
              <a href="#como-funciona" className="btn-secondary">
                Como Funciona
              </a>
            </div>

            {/* Vehicle Types with Images */}
            <div className="grid grid-cols-3 gap-4 mt-8">
              <motion.div
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: 0.5 }}
                className="group relative overflow-hidden rounded-xl border border-[var(--border-subtle)] hover:border-[var(--accent-primary)] transition-all duration-300"
              >
                <div className="aspect-[4/3] overflow-hidden">
                  <img 
                    src={vehicleImages.car} 
                    alt="Carro" 
                    className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500"
                  />
                  <div className="absolute inset-0 bg-gradient-to-t from-[var(--bg-primary)] via-transparent to-transparent" />
                </div>
                <div className="absolute bottom-0 left-0 right-0 p-3 flex items-center gap-2">
                  <Car size={20} className="text-[var(--accent-primary)]" />
                  <span className="font-semibold text-sm">Carros</span>
                </div>
              </motion.div>

              <motion.div
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: 0.6 }}
                className="group relative overflow-hidden rounded-xl border border-[var(--border-subtle)] hover:border-[var(--accent-primary)] transition-all duration-300"
              >
                <div className="aspect-[4/3] overflow-hidden">
                  <img 
                    src={vehicleImages.moto} 
                    alt="Moto" 
                    className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500"
                  />
                  <div className="absolute inset-0 bg-gradient-to-t from-[var(--bg-primary)] via-transparent to-transparent" />
                </div>
                <div className="absolute bottom-0 left-0 right-0 p-3 flex items-center gap-2">
                  <Bike size={20} className="text-[var(--accent-primary)]" />
                  <span className="font-semibold text-sm">Motos</span>
                </div>
              </motion.div>

              <motion.div
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: 0.7 }}
                className="group relative overflow-hidden rounded-xl border border-[var(--border-subtle)] hover:border-[var(--accent-primary)] transition-all duration-300"
              >
                <div className="aspect-[4/3] overflow-hidden">
                  <img 
                    src={vehicleImages.truck} 
                    alt="Caminhão" 
                    className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500"
                  />
                  <div className="absolute inset-0 bg-gradient-to-t from-[var(--bg-primary)] via-transparent to-transparent" />
                </div>
                <div className="absolute bottom-0 left-0 right-0 p-3 flex items-center gap-2">
                  <Truck size={20} className="text-[var(--accent-primary)]" />
                  <span className="font-semibold text-sm">Caminhões</span>
                </div>
              </motion.div>
            </div>
          </motion.div>

          {/* Right Content - Visual */}
          <motion.div
            initial={{ opacity: 0, x: 50 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.7, delay: 0.3 }}
            className="relative hidden lg:block"
          >
            {/* Main Visual Container */}
            <div className="relative">
              {/* Phone Mockup */}
              <div className="relative w-72 h-[500px] mx-auto">
                {/* Phone Frame */}
                <div className="absolute inset-0 bg-gradient-to-b from-[var(--bg-tertiary)] to-[var(--bg-secondary)] rounded-[40px] border-4 border-[var(--border-primary)] shadow-2xl">
                  {/* Screen */}
                  <div className="absolute inset-4 bg-[var(--bg-primary)] rounded-[28px] overflow-hidden">
                    {/* App Header */}
                    <div className="bg-[var(--bg-secondary)] p-4 border-b border-[var(--border-subtle)]">
                      <div className="flex items-center gap-3">
                        <div className="w-10 h-10 rounded-full bg-[var(--accent-bg)] flex items-center justify-center">
                          <MapPin size={20} className="text-[var(--accent-primary)]" />
                        </div>
                        <div>
                          <p className="text-sm font-semibold">Meu Veículo</p>
                          <p className="text-xs text-[var(--accent-primary)]">Online</p>
                        </div>
                      </div>
                    </div>
                    
                    {/* Map Area */}
                    <div className="relative h-48 bg-[var(--bg-tertiary)]">
                      {/* Simulated Map Grid */}
                      <div 
                        className="absolute inset-0 opacity-20"
                        style={{
                          backgroundImage: `linear-gradient(var(--border-primary) 1px, transparent 1px), linear-gradient(90deg, var(--border-primary) 1px, transparent 1px)`,
                          backgroundSize: '20px 20px'
                        }}
                      />
                      {/* Location Pin */}
                      <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2">
                        <div className="relative">
                          <div className="w-4 h-4 bg-[var(--accent-primary)] rounded-full animate-ping absolute" />
                          <div className="w-4 h-4 bg-[var(--accent-primary)] rounded-full relative z-10" />
                        </div>
                      </div>
                    </div>
                    
                    {/* Info Cards */}
                    <div className="p-4 space-y-3">
                      <div className="bg-[var(--bg-secondary)] rounded-xl p-3 flex items-center justify-between">
                        <div className="flex items-center gap-2">
                          <Shield size={18} className="text-[var(--accent-primary)]" />
                          <span className="text-sm">Status</span>
                        </div>
                        <span className="text-xs text-[var(--accent-primary)] font-medium">Protegido</span>
                      </div>
                      <div className="bg-[var(--bg-secondary)] rounded-xl p-3 flex items-center justify-between">
                        <div className="flex items-center gap-2">
                          <MapPin size={18} className="text-[var(--accent-cyan)]" />
                          <span className="text-sm">Localização</span>
                        </div>
                        <span className="text-xs text-[var(--text-muted)]">Belém - PA</span>
                      </div>
                    </div>
                  </div>
                </div>
                
                {/* Notch */}
                <div className="absolute top-4 left-1/2 transform -translate-x-1/2 w-20 h-6 bg-[var(--bg-primary)] rounded-full" />
              </div>

              {/* Floating Cards */}
              <motion.div
                animate={{ y: [0, -10, 0] }}
                transition={{ duration: 3, repeat: Infinity, ease: "easeInOut" }}
                className="absolute -left-8 top-20 bg-[var(--bg-secondary)] border border-[var(--border-subtle)] rounded-xl p-4 shadow-xl"
              >
                <div className="flex items-center gap-3">
                  <div className="w-10 h-10 rounded-lg bg-[var(--accent-bg)] flex items-center justify-center">
                    <Shield size={20} className="text-[var(--accent-primary)]" />
                  </div>
                  <div>
                    <p className="text-xs text-[var(--text-muted)]">Bloqueio</p>
                    <p className="text-sm font-semibold text-[var(--accent-primary)]">Ativo</p>
                  </div>
                </div>
              </motion.div>

              <motion.div
                animate={{ y: [0, 10, 0] }}
                transition={{ duration: 3, repeat: Infinity, ease: "easeInOut", delay: 1.5 }}
                className="absolute -right-8 bottom-32 bg-[var(--bg-secondary)] border border-[var(--border-subtle)] rounded-xl p-4 shadow-xl"
              >
                <div className="flex items-center gap-3">
                  <div className="w-10 h-10 rounded-lg bg-[var(--accent-cyan-bg)] flex items-center justify-center">
                    <Smartphone size={20} className="text-[var(--accent-cyan)]" />
                  </div>
                  <div>
                    <p className="text-xs text-[var(--text-muted)]">App</p>
                    <p className="text-sm font-semibold">Controle Total</p>
                  </div>
                </div>
              </motion.div>
            </div>
          </motion.div>
        </div>
      </div>
    </section>
  );
};

export default Hero;
